unit IWGrids;
{PUBDIST}

interface

uses
  {$IFDEF VSNET}System.ComponentModel, System.ComponentModel.Design,
  System.Drawing, System.Drawing.Design, System.Collections, System.Data,
  IWNETBaseControl, {$ENDIF}
  {$IFDEF Linux}QGraphics, {$ELSE}Graphics, {$ENDIF}
  Classes,
  {$IFDEF VSNET}
  IWNetComponent,IWGridCellConverter, Atozed.Intraweb.DB,
  {$ELSE}
    {$IFDEF Linux}
    IWCLXComponent, IWCLXClasses,
    {$ELSE}
    IWVCLComponent, IWVCLClasses,
    {$ENDIF}
  {$IFDEF Linux}QControls, {$ELSE}Controls, {$ENDIF}
  {$ENDIF}
  IWGridCommon, IWControl, IWFont, IWTypes, IWHTMLTag, IWColor, IWApplication,
  IWRenderContext, IWBaseHTMLInterfaces, IWHTML40Interfaces, IWContainer,
  IWBaseRenderContext, IWContainerLayout, IWBaseInterfaces;

type
{$IFDEF VSNET}
  TIWGridColumn = class
  public
    Name : String;
    Header : String;
  end;
{$ENDIF}


  TIWGridCell = class;
  TIWCustomGrid = class;
  // This is a TCollectionItem so in the future it can be made visual easier later.

  {$IFDEF VSNET}
  [TypeConverterAttribute(typeof(TIWGridCellConverter))]
  {$ENDIF}
  TIWGridCell = class(TCollectionItem)
  //COLSPAN=Number (columns spanned by the cell)
  //ROWSPAN=Number (rows spanned by the cell)
  private
    FGrid: TIWCustomGrid;
  protected
    FDoRefreshControl: Boolean;
    FAlignment: TAlignment;
    FBGColor: TIWColor;
    FClickable: Boolean;
    FControl: TIWCustomControl;
    FDesignMode: Boolean;
    FDoSubmitValidation: Boolean;
    FFont: TIWFont;
    FHeader: Boolean;
    FHeight: string;
    FTag: TObject;
    FText: string;
    FVAlign: TIWGridVAlign;
    FVisible: Boolean;
    FWidth: string;
    FWrap: Boolean;
    FHint: string;
    FRawText : Boolean;
    //
    procedure SetControl(const AValue: TIWCustomControl);
    procedure SetGrid(const AValue: TIWCustomGrid);
    procedure SetFont(const AValue: TIWFont);
    procedure SetHeight(const AValue: string);
    procedure SetWidth(const AValue: string);

    procedure AssignTo(ADest: TPersistent); override;

    property DesignMode: Boolean read FDesignMode;
  public
    constructor Create(ACollection: TCollection); overload; override;
  {$IFDEF VSNET}
    constructor Create(
		    AAlignment: TAlignment;
		    ABGColor: TIWColor;
		    AControl: TIWCustomControl;
		    ADoSubmitValidation: Boolean;
        AWebFont: TIWFont;
		    AHeader: Boolean;
		    AHeight: string;
		    AHint: string;
		    ATag: TObject;
		    AText: string;
		    AVAlign: TIWGridVAlign;
		    AVisible: Boolean;
		    AWidth: string;
		    AWrap: Boolean
		    ); reintroduce; overload;     
    {$ENDIF}
    destructor Destroy; override;

    function RenderSelf(AGrid: TIWCustomGrid; const ARow: Integer;
      const AColumn: Integer; AContext: TIWComponent40Context; AText: string = ''): TIWHTMLTag; virtual;
    procedure Invalidate;
    //
    property Grid: TIWCustomGrid read FGrid write SetGrid;
    property DoRefreshControl: Boolean read FDoRefreshControl write FDoRefreshControl;
    property Clickable: Boolean read FClickable write FClickable;
    {$IFDEF CLR}
    property Font: TIWFont read FFont write SetFont;
    {$ENDIF}
  published
    property Alignment: TAlignment read FAlignment write FAlignment;
    property BGColor: TIWColor read FBGColor write FBGColor;
    property Control: TIWCustomControl read FControl write SetControl;
    property DoSubmitValidation: Boolean read FDoSubmitValidation write FDoSubmitValidation;
    property {$IFDEF CLR}WebFont{$ELSE}Font{$ENDIF}: TIWFont read FFont write SetFont;
    property Header: Boolean read FHeader write FHeader;
    property Height: string read FHeight write SetHeight;
    property Hint: string read FHint write FHint;
    property Tag: TObject read FTag write FTag;
    property Text: string read FText write FText;
    property VAlign: TIWGridVAlign read FVAlign write FVAlign;
    property Visible: Boolean read FVisible write FVisible;
    property Width: string read FWidth write SetWidth;
    property Wrap: Boolean read FWrap write FWrap;
    property RawText : Boolean read FRawText write FRawText;
  end;

  TIWGridCells = class(TOwnedCollection)
  protected
    FDoRefreshControl: Boolean;
  public
    function Add: TIWGridCell;
    // All this is for compatibility with Delphi 5 because Owner property is not defined in
    // TOWnedCollection
    function Owner: TPersistent;
    procedure Invalidate;

    property  DoRefreshControl: Boolean read FDoRefreshControl write FDoRefreshControl;
  end;

  TIWOnRenderCell = procedure(ACell: TIWGridCell; const ARow: Integer; const AColumn: Integer)
    of object;

  //TODO: If we have more container control combos, make a TIWCustomControlContainer that
  TIWCustomGrid = class(TIWCustomControl, IIWBaseContainer, IIWHTML40Container, IIWFreeNotification)
  private
    FContainerImplementation: TIWContainerImplementation;
    FOnRender: TNotifyEvent;
  protected
    //Align is not supported because Grid is abosolutely positioned. Align has also been depricated
    //in HTML 4.
    FBorderColors: TIWGridBorderColors;
    FBorderSize: Integer;
    FBorderStyle: TIWGridBorderStyle;
    FBGColor: TIWColor;
    FCells: array of array of TIWGridCell;
    FCellPadding: Integer;
    FCellSpacing: Integer;
    FFrameBuffer: Integer;
    FLines: TIWGridLines;
    FOnRenderCell: TIWOnRenderCell;
    FSummary: string;
    FUseSize: Boolean;
    //
    {$IFDEF VSNET}
    procedure DataBind; virtual;
    function getText: string; reintroduce;
    procedure setText(AValue: string); reintroduce;
    {$ENDIF}
    procedure DoRender; virtual;
    procedure DoRenderCell(ACell: TIWGridCell; const ARow: Integer; const AColumn: Integer);
    procedure RenderCells(AContext: TIWComponent40Context; AGridTag: TIWHTMLTag); virtual;
    function IsValidCell(ACell: TIWGridCell): Boolean; virtual; abstract;

    procedure InitControl; override;

    // Container functions
    function ContainerPrefix: string; 
    function ContainerName: string;
    function ContainerClassname: string;
    function InterfaceInstance: TPlatformComponent;
    procedure IWAddComponent(AComponent: TPlatformComponent);
    procedure IWRemoveComponent(AComponent: TPlatformComponent);
    function IWFindComponent(AComponent: TPlatformComponent): Integer;
    function FindComponentByName(AComponent: String): TPlatformComponent;
    function get_Component(AIndex: Integer): TPlatformComponent;
    function get_IWComponentsCount: Integer;
    function InitContainerContext(AWebApplication: TIWApplication): TIWContainerContext;

    procedure RenderComponents(AContainerContext: TIWContainerContext;
      APageContext: TIWBasePageContext);
    procedure RenderHTMLComponents(AContainerContext: TIWContainerContext;
      APageContext: TIWBaseHTMLPageContext);

    procedure SetActiveControl(AControl: IIWHTML40Control);

    procedure set_LayoutMgr(Value: TIWContainerLayout);
    function get_LayoutMgr: TIWContainerLayout;

    function get_ContainerContext: TIWContainerContext;
    procedure set_ContainerContext(const AContainerContext: TIWContainerContext);

    function get_HasTabOrder: boolean;override;

    function GenerateControlPositions: String;
    function CheckComponentForRender(AComponent: TPlatformComponent): Boolean;

    procedure Freeing(AObject: TObject); virtual;

    property LayoutMgr: TIWContainerLayout read get_LayoutMgr write set_LayoutMgr;
    property ContainerContext: TIWContainerContext read get_ContainerContext write set_ContainerContext;
    property Component[AIndex: Integer]: TPlatformComponent read get_Component;

    property IWComponentsCount: Integer read get_IWComponentsCount;
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  public
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
  published
    {$IFDEF VSNET}[DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]{$ENDIF}
    property BorderColors: TIWGridBorderColors read FBorderColors write FBorderColors;
    property BGColor: TIWColor read FBGColor write FBGColor;
    property BorderSize: Integer read FBorderSize write FBorderSize;
    property BorderStyle: TIWGridBorderStyle read FBorderStyle write FBorderStyle;
    {$IFDEF VSNET}
    property Text: string read getText write setText;
    {$ELSE}
    property Caption;
    {$ENDIF}
    property CellPadding: Integer read FCellPadding write FCellPadding;
    property CellSpacing: Integer read FCellSpacing write FCellSpacing;
    {$IFDEF CLR}
    property WebFont;
    {$ELSE}
    property Font;
    {$ENDIF}
    property FrameBuffer: Integer read FFrameBuffer write FFrameBuffer;
    property Lines: TIWGridLines read FLines write FLines;
    property OnRenderCell: TIWOnRenderCell read FOnRenderCell write FOnRenderCell;
    {$IFDEF VSNET} [DefaultValue('')] {$ENDIF}
    property Summary: string read FSummary write FSummary;
    property UseFrame;
    property UseSize: Boolean read FUseSize write FUseSize;
    //
    property OnRender: TNotifyEvent read FOnRender write FOnRender;
  end;

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWGrid.bmp}
  TIWGrid = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWGrid), 'TIWGrid.bmp')]
  {$ENDIF}
  TIWGrid = class(TIWCustomGrid, IIWSubmitControl)
  protected
    FShowEmptyCells : Boolean;
    FSubmitParam : String;
    FCellCollection: TIWGridCells;
    FColumnCount: Integer;
    FOnCellClick: TIWOnCellClick;
    FRowCount: Integer;
    FCurrentRow : Integer;
    FScrollToCurrentRow : Boolean;

    {$IFDEF VSNET}
    FView : DataView;
    FDataSource : TObject;
    FDataMember : string;
    FColumns : ArrayList;
    procedure CreateColumns;
    procedure DataBind; override;
    {$ENDIF}
    //
    procedure DoCellClick(const ARow: Integer; const AColumn: Integer); virtual;
    function GetCell(const ARow, AColumn: Integer): TIWGridCell;
    procedure RenderCells(AContext: TIWComponent40Context; AGridTag: TIWHTMLTag); override;
    procedure SetColumnCount(const AValue: Integer);
    procedure SetRowCount(const AValue: Integer);
    procedure Submit(const AValue: string); override;
    function IsValidCell(ACell: TIWGridCell): Boolean; override;

    procedure Freeing(AObject: TObject); override;

    procedure InitControl; override;
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  public
    procedure Clear;
    function GetSubmitParam : String;

    function CellExists(const ARow, AColumn: Integer): Boolean;
    procedure DeleteColumn(const AColumn: Integer);
    procedure DeleteRow(const ARow: Integer);

    property Cell[const ARow: Integer; const AColumn: Integer]: TIWGridCell read GetCell;
    property CurrentRow : Integer read FCurrentRow write FCurrentRow;
  published
    property FriendlyName;
    property ColumnCount: Integer read FColumnCount write SetColumnCount;
    property OnCellClick: TIWOnCellClick read FOnCellClick write FOnCellClick;
    property RowCount: Integer read FRowCount write SetRowCount;
    property ShowEmptyCells : Boolean read FShowEmptyCells write FShowEmptyCells;
    property ScrollToCurrentRow : Boolean read FScrollToCurrentRow write FScrollToCurrentRow;
    {$IFDEF VSNET}
    [Category('Data')]
    [Editor('Atozed.Intraweb.Design.DataSourcePropertyEditor, Atozed.Intraweb.Design',typeof(UITypeEditor))]
    property DataSource: TObject read FDataSource write FDataSource;
    [Category('Data')]
    [Editor('Atozed.Intraweb.Design.DataMemberPropertyEditor, Atozed.Intraweb.Design',typeof(UITypeEditor))]
    property DataMember: string read FDataMember write FDataMember;
    {$ENDIF}
  end;

implementation

uses
  SysUtils,
  IWForm, IWAppForm, IWServerControllerBase, IWResourceStrings,
  IWBaseControl,
  SWSystem, SWStrings, IWMarkupLanguageTag, IWStreams, IWBaseHTMLControl;

{ TIWGridCell }

procedure TIWGridCell.AssignTo(ADest: TPersistent);
begin
  if ADest is TIWGridCell then begin
    with TIWGridCell(ADest) do begin
      Alignment := Self.Alignment;
      BGColor := Self.BGColor;
      Font.Assign(Self.Font);
      Header := Self.Header;
      Height := Self.Height;
      Text := Self.Text;
      VAlign := Self.VAlign;
      Width := Self.Width;
      Wrap := Self.Wrap;
      Visible := Self.Visible;
      Control := Self.Control;
    end;
  end else begin
    inherited;
  end;
end;

constructor TIWGridCell.Create(ACollection: TCollection);
begin
  inherited Create(ACollection);
  FBGColor := fromTColor(clNone);
  FDoSubmitValidation := True;
  {$IFDEF VSNET}
  FFont := Atozed.Intraweb.IWFont.TIWFont.Create;
  {$ELSE}
  FFont := TIWFont.Create;
  {$ENDIF}
  // Copy from Grids font.
  FFont.Assign(Font);
  FHeight := '0';
  FVisible := True;
  FWidth := '0';
  FControl := nil;
  FRawText := True;
end;

{$IFDEF VSNET}
constructor TIWGridCell.Create(
    AAlignment: TAlignment;
    ABGColor: TIWColor;
    AControl: TIWCustomControl;
    ADoSubmitValidation: Boolean;
    AWebFont: TIWFont;
    AHeader: Boolean;
    AHeight: string;
    AHint: string;
    ATag: TObject;
    AText: string;
    AVAlign: TIWGridVAlign;
    AVisible: Boolean;
    AWidth: string;
    AWrap: Boolean
    );
begin
  Create(nil);
  Alignment:= AAlignment;
  BGColor:= ABGColor;
  Control:= AControl;
  DoSubmitValidation:= ADoSubmitValidation;
  WebFont:= AWebFont;
  Header:= AHeader;
  Height:= AHeight;
  Hint:= AHint;
  Tag:= ATag;
  Text:= AText;
  VAlign:= AVAlign;
  Visible:= AVisible;
  Width:= AWidth;
  Wrap:= AWrap;
end;
{$ENDIF}

destructor TIWGridCell.Destroy;
begin
  FreeAndNil(FFont);
  Control := nil;
  {if Assigned(FControl) and not Assigned(FControl.Owner) then begin
    FreeAndNil(FControl);
  end;}
  inherited Destroy;
end;

procedure TIWCustomGrid.DoRenderCell(ACell: TIWGridCell; const ARow, AColumn: Integer);
begin
  if Assigned(OnRenderCell) then begin
    OnRenderCell(ACell, ARow, AColumn);
  end;
end;

procedure TIWGridCell.Invalidate;
begin
  DoRefreshControl := True;
  if Assigned(FGrid) then begin
    FGrid.Invalidate;
  end else begin
    if Assigned(Collection) and (Collection is TIWGridCells) then
      (Collection as TIWGridCells).Invalidate;
  end;
end;

function TIWGridCell.RenderSelf(AGrid: TIWCustomGrid; const ARow: Integer;
  const AColumn: Integer; AContext: TIWComponent40Context; AText: string = ''): TIWHTMLTag;
var
  LControlRenderData: TIWHTMLTag;
  LVAlign: string;
  LTmp: String;
  LControlContext: TIWComponent40Context;
  LContainerContext: TIWContainerContext;
begin
  if not Visible then begin
    Result := TIWHTMLTag.CreateTag('TD'); try
      Result.Contents.AddText('&nbsp;');
    except FreeAndNil(Result); raise; end;
  end else begin
    LContainerContext := HTML40ContainerInterface(AGrid).ContainerContext;
    AGrid.DoRenderCell(Self, ARow, AColumn);
    LControlRenderData := nil;
    LControlContext := nil;
    if Control <> nil then begin
      Control.Visible := true;
      LControlContext := TIWComponent40Context.Create(Control, LContainerContext, AContext.PageContext);
      LControlRenderData := Control.RenderHTML(LControlContext);
      {$IFDEF CLR}
      if Control.WebFont.CSSStyle<>'' then begin
        LControlRenderData.AddStringParam('CLASS',Control.WebFont.CSSStyle);
      end;
      {$ELSE}
      if Control.Font.CSSStyle<>'' then begin
        LControlRenderData.AddStringParam('CLASS',Control.Font.CSSStyle);
      end;
      {$ENDIF}
      LControlRenderData.AddParmsList(Control.ExtraTagParams);
      LControlContext.HTMLTag := LControlRenderData;

      if Assigned(LControlRenderData) then begin
        HTML40ControlInterface(Control).DoHTMLTag(LControlRenderData);
        LTmp := Control.RenderStyle(LControlContext);
        if LControlRenderData.Params.Values['STYLE'] <> '' then begin
          LTmp := LTmp + ' ' + LControlRenderData.Params.Values['STYLE'] + ';';
        end;
        if LControlRenderData.Params.Values['ID'] = '' then begin
          LControlRenderData.AddStringParam('ID', Control.HTMLName);
        end;
        if LControlRenderData.Params.Values['NAME'] = '' then begin
          LControlRenderData.AddStringParam('NAME', Control.HTMLName);
        end;
        {if (LControlRenderData.Params.Values['CLASS'] = '') then begin
          LControlRenderData.AddStringParam('CLASS', Control.RenderCSSClass(LControlContext));
        end;}
        // if not AnsiSameText(Control.RenderCSSClass(LControlContext), Control.HTMLName + 'CSS') then begin
          LControlRenderData.AddStringParam('STYLE', LTmp);
        (*end else begin
          TIWPageContext40(AContext.PageContext).StyleTag.Contents.AddText('.' + Control.HTMLName + 'CSS { ' + LTmp + '}' + EOL)
        end;*)
        if SupportsInterface(Control, IIWHTML40Component) then begin
          HTML40ComponentInterface(Control).RenderScripts(LControlContext);
        end;
      end;
      LContainerContext.AddComponent(LControlContext);
      DoRefreshControl := DoRefreshControl or Control.DoRefreshControl;
      if DoRefreshControl then begin
        Invalidate;
      end;
    end;
    // VAlign
    LVAlign := '';
    case FVAlign of
      vaMiddle: LVAlign := 'VAlign=middle';
      vaTop: LVAlign := 'VAlign=top';
      vaBottom: LVAlign := 'VAlign=bottom';
      vaBaseline: LVAlign := 'VAlign=baseline';
    end;
    //
    if Length(AText) = 0 then begin
      AText := Text;
    end;

    if not RawText then begin
      AText := TIWBaseHTMLControl.TextToHTML(AText, true, false);
    end;

    AText := SWSystem.iif((Trim(AText) = '') and (LControlRenderData = nil), '&nbsp;', AText);

    Result := TIWHTMLTag.CreateTag(SWSystem.iif(Header, 'TH', 'TD')); try
      Result.Add(LVAlign);
      // Always render alignment, default is different or TH and TD
      Result.AddStringParam('align', IWGridAlignments[Alignment]);
      if Trim(FHint) <> '' then begin
        Result.AddStringParam('title', FHint);
      end;
      Result.Add(SWSystem.iif(not Wrap, 'NOWRAP'));
      Result.Add(SWSystem.iif((Width <> '') and (Width <> '0'), 'Width=' + Width));
      Result.Add(SWSystem.iif((Height <> '') and (Height <> '0'), 'Height=' + Height));
      Result.Add(HTMLColor(BGColor, 'BGColor'));
      {$IFDEF CLR}
      if Length(WebFont.CSSStyle) > 0 then begin
        Result.Add('CLASS=' + WebFont.CSSStyle);
      {$ELSE}
      if Length(Font.CSSStyle) > 0 then begin
        Result.Add('CLASS=' + Font.CSSStyle);
      {$ENDIF}
        Result.AddStringParam('style','position:static;');
      end;

      // FontToString checks Font.Enabled
      if Trim(AText) <> '' then begin
        if Font.CSSStyle = '' then begin
          if Result.Contents.AddTagAsObject(Font.FontToStringTag(AText)) = nil then begin
            Result.Contents.AddText(AText);
          end;
        end else begin
          Result.Contents.AddText(AText);
        end;
      end;

      if Assigned(LControlRenderData) then begin
        Result.Contents.AddTagAsObject(LControlRenderData);
        LControlContext.HTMLTag := nil;
      end;
    except
      FreeAndNil(Result);
      raise;
    end;
  end;
end;

procedure TIWGridCell.SetGrid(const AValue: TIWCustomGrid);
begin
  FGrid := AValue;
end;

procedure TIWGridCell.SetControl(const AValue: TIWCustomControl);
begin
  if AValue <> FControl then begin
    if Assigned(FControl) then begin
      BaseControlInterface(FControl).ParentChanging(FGrid, nil);
    end;
  end;
  if AValue <> nil then begin
    if not AValue.IsDesignMode then begin
      AValue.Visible := False;
      // Not changing the control's actual parent will trick the form to render SetFocus code
      BaseControlInterface(AValue).ParentChanging(AValue.Parent, FGrid);
    end;

    if AValue is TIWCustomGrid then begin
      if FGrid = AValue then begin
        raise Exception.Create(RSCanSetControlToSelf);
      end;
      with TIWCustomGrid(AValue) do begin
        if IsValidCell(self) then begin
          raise Exception.Create(RSCanSetControlToSelf);
        end;
      end;
    end;
  end;
  FControl := AValue;
  Invalidate;
end;

procedure TIWGridCell.SetFont(const AValue: TIWFont);
begin
  FFont.Assign(AValue);
  Invalidate;
end;

procedure TIWGridCell.SetHeight(const AValue: string);
var
  s: string;

  procedure InvalidHeight;
  begin
    raise Exception.Create(Format(RSInvalidHeight, [s]));
  end;

begin
  s := Trim(AValue);
  if Length(s) = 0 then begin
    InvalidHeight;
  // Percentage must be >0
  end else if (s[Length(s)] = '%') and (StrToIntDef(Copy(s, 1, Length(s) - 1), 0) > 0) then begin
    FHeight := AValue;
  end else if StrToIntDef(s, -1) >= 0 then begin
    FHeight := AValue;
  end else begin
    InvalidHeight;
  end;
  Invalidate;
end;

procedure TIWGridCell.SetWidth(const AValue: string);
var
  s: string;

  procedure InvalidWidth;
  begin
    raise Exception.Create(Format(RSInvalidWidth, [s]));
  end;

begin
  s := Trim(AValue);
  if Length(s) = 0 then begin
    InvalidWidth;
  // Percentage must be >0
  end else begin
    if ( s[Length(s)] = '%') and (StrToIntDef(Copy(s, 1, Length(s) - 1), 0) > 0 ) or
        ( s[Length(s)] = '*') then begin
      FWidth := AValue;
    end else begin
      if StrToIntDef(s, -1) >= 0 then begin
        FWidth := AValue;
      end else begin
        InvalidWidth;
      end;
    end;
  end;
  Invalidate;
end;

{ TIWCustomGrid }

function TIWCustomGrid.CheckComponentForRender(AComponent: TPlatformComponent): Boolean;
begin
  if SupportsInterface(Parent, IIWHTML40Form) then begin
    result := HTML40FormInterface(Parent).PageContext.UpdateMode = umPartial;
  end else begin
    if SupportsInterface(Parent, IIWHTML40Container) then begin
      result := HTML40ContainerInterface(Parent).CheckComponentForRender(AComponent);
    end else begin
      result := false;
    end;
  end;
end;

procedure TIWCustomGrid.InitControl;
begin
  FContainerImplementation := TIWContainerImplementation.Create(self);
  inherited;
  FBorderColors := TIWGridBorderColors.Create;
  FBGColor := fromTColor(clNone);
  FBorderSize := 1;
  FFrameBuffer := 40;
  FNeedsFormTag := True;
  FUseSize := True;
  FSummary := '';
  Width := 300;
  Height := 150;
end;

{$IFDEF VSNET}

procedure TIWCustomGrid.DataBind;
begin
  // empty
end;


procedure TIWGrid.DataBind;
var
  i,j : Integer;
  LColumnCount : Integer;
  LColumn : TIWGridColumn;
  LDatum : System.Object;
begin
  if FDataSource = nil then begin
    Exit;
  end;

  CreateColumns;

  LColumnCount := FColumns.Count;

  // set grid dimenstions
  ColumnCount := LColumnCount;
  RowCount := FView.Count + 1;

  for i := 0 to LColumnCount - 1 do begin
    LColumn := FColumns[i] as TIWGridColumn;

    for j := 0 to RowCount - 1 do begin
      if j = 0 then begin
        // it's a header cell
        Cell[j,i].Text := LColumn.Header;
        Cell[j,i].Header := true;
      end else begin
        // it's a data ce;;
        LDatum := FView[j-1][LColumn.Name];
        Cell[j,i].Text := LDatum.ToString();
      end;


    end;
  end;


end;


procedure TIWGrid.CreateColumns;
var
  i : Integer;
  LColumnCount : Integer;
  LTable : DataTable;
  LDataColumn : DataColumn;
  LGridColumn : TIWGridColumn;
begin
  if FDataSource <> nil then begin
    // det the DataTable and DataView
    FColumns := ArrayList.Create;
    if DataSource is DataTable then begin
      LTable := DataSource as DataTable;
      FView := LTable.DefaultView;
    end else begin
      if DataSource is DataSet then begin
        LTable := (DataSource as DataSet).Tables[DataMember];
        FView := LTable.DefaultView;
      end else begin
        if DataSource is DataView then begin
          FView := DataSource as DataView;
          LTable := FView.Table;
        end;
      end;
    end;

    LColumnCount := LTable.Columns.Count;
    for i := 0 to LColumnCount - 1 do begin
      LDataColumn := LTable.Columns[i];

      LGridColumn := TIWGridColumn.Create;
      LGridColumn.Name := LDataColumn.ColumnName;
      LGridColumn.Header := LDataColumn.ColumnName;
      FColumns.Add(LGridColumn);
    end;
  end;
end;
{$ENDIF}


function TIWCustomGrid.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
begin
{$IFDEF VSNET}
  DataBind;
{$ENDIF}
  // If Control property is used.
  Result := TIWHTMLTag.CreateTag('TABLE');  try
    // HTMLName is already used by the surrounding DIV as per BBG #1003
    if UseFrame then begin
      Result.AddStringParam('ID', 'tbl' + HTMLName);
    end else begin
      Result.AddStringParam('ID', HTMLName);
    end;


    if BorderSize >= 0 then begin
      Result.AddIntegerParam('Border', BorderSize);
    end;
    Result.AddColor('BGColor', BGColor);
    Result.AddIntegerParam('CELLPADDING', CellPadding);
    Result.AddIntegerParam('CELLSPACING', CellSpacing);

    if UseSize then begin
      if UseFrame then begin
        Result.AddStringParam('WIDTH','100%');
      end else begin
        Result.AddIntegerParam('WIDTH',Width);
      end;
    end;
    
    set_RenderSize(UseSize);
    if BorderSize > 0 then begin
      Result.AddColor('BORDERCOLOR', BorderColors.Color);
      Result.AddColor('BORDERCOLORLIGHT', BorderColors.Light);
      Result.AddColor('BORDERCOLORDARK', BorderColors.Dark);

      case Lines of
        tlNone: Result.AddStringParam('RULES', 'none');
        tlRows: Result.AddStringParam('RULES', 'rows');
        tlCols: Result.AddStringParam('RULES', 'cols');
      end;

      case BorderStyle of
        tfVoid: Result.AddStringParam('frame', 'void');
        tfAbove: Result.AddStringParam('frame', 'above');
        tfBelow: Result.AddStringParam('frame', 'below');
        tvHorizSides: Result.AddStringParam('frame', 'hsides');
        tfVertSides: Result.AddStringParam('frame', 'vsides');
        tfLeft: Result.AddStringParam('frame', 'lhs');
        tfRight: Result.AddStringParam('frame', 'rhs');
      end;
    end;
    Result.Contents.AddText(iif(Caption, '<CAPTION>' + Caption + '</CAPTION>'));
    Result.Contents.AddText(iif(Summary, '<SUMMARY>' + Summary + '</SUMMARY>'));
    RenderCells(TIWComponent40Context(AContext), Result);
  except FreeAndNil(Result); raise; end;
end;

procedure TIWCustomGrid.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FBorderColors);
  inherited;
  FreeAndNil(FContainerImplementation);
end;

// IIWContainer methods

procedure TIWCustomGrid.IWAddComponent(AComponent: TPlatformComponent);
begin
  FContainerImplementation.AddComponent(AComponent);
end;

function TIWCustomGrid.ContainerName: string;
begin
  result := FContainerImplementation.ContainerName;
end;

function TIWCustomGrid.ContainerClassname: string;
begin
  Result := FContainerImplementation.ContainerClassname;
end;

function TIWCustomGrid.IWFindComponent(AComponent: TPlatformComponent): Integer;
begin
  result := FContainerImplementation.FindComponent(AComponent);
end;

function TIWCustomGrid.FindComponentByName(AComponent: String): TPlatformComponent;
begin
  result := FContainerImplementation.FindComponentByName(AComponent);
end;

function TIWCustomGrid.get_Component(AIndex: Integer): TPlatformComponent;
begin
  result := FContainerImplementation.GetComponent(AIndex);
end;

function TIWCustomGrid.get_IWComponentsCount: Integer;
begin
  result := FContainerImplementation.get_IWComponentsCount;
end;

function TIWCustomGrid.InitContainerContext(AWebApplication: TIWApplication): TIWContainerContext;
begin
  result := ContainerContext;
  ContainerContext.LayoutManager := TIWContainerLayout.Create(Self);
  TIWContainerLayout(ContainerContext.LayoutManager).SetContainer(Self);
end;

function TIWCustomGrid.InterfaceInstance: TPlatformComponent;
begin
  result := Self;
end;

procedure TIWCustomGrid.IWRemoveComponent(AComponent: TPlatformComponent);
begin
  FContainerImplementation.RemoveComponent(AComponent);
end;

procedure TIWCustomGrid.RenderComponents(AContainerContext: TIWContainerContext;
  APageContext: TIWBasePageContext);
begin
  RenderHTMLComponents(AContainerContext, TIWBaseHTMLPageContext(APageContext));
end;

procedure TIWCustomGrid.RenderHTMLComponents(AContainerContext: TIWContainerContext;
  APageContext: TIWBaseHTMLPageContext);
Var
  i: integer;
  LContainerContext: IWBaseRenderContext.TIWContainerContext;
  LIWContainer: IIWBaseContainer;
begin
  // Process controls to add Partial update SPAN tags.
  // Componenets are rendered during RenderHTML inside the coresponding cells
  with TIWContainerLayout(ContainerContext.LayoutManager) do begin
    ProcessControls(ContainerContext, APageContext);

    // Becouse HTML Tags are already embeded into table tag we should clear the HTML pointer inside container context.
    with ContainerContext do begin
      for i := 0 to ComponentsCount - 1 do begin
        if SupportsInterface(ComponentsList[i], IIWBaseContainer) then begin
          LIWContainer := BaseContainerInterface(ComponentsList[i]);
          LContainerContext := LIWContainer.InitContainerContext(APageContext.WebApplication);
          if Assigned(LContainerContext) then begin
            LContainerContext.UpdateTree := {CheckUpdateTree(LComponent) or }AContainerContext.UpdateTree;
            LIWContainer.RenderComponents(LContainerContext, APageContext);
          end;
        end;

        TIWBaseHTMLComponentContext(ComponentContext[BaseHTMLComponentInterface(ComponentsList[i]).HTMLName]).HTMLTag := nil;
      end;
    end;

    // SetContainer(nil);
    Free;
  end;
end;


procedure TIWCustomGrid.SetActiveControl(AControl: IIWHTML40Control);
begin
  //
end;

procedure TIWCustomGrid.set_LayoutMgr(Value: TIWContainerLayout);
begin
  //
end;

function TIWCustomGrid.get_LayoutMgr: TIWContainerLayout;
begin
  result := nil;
end;

function TIWCustomGrid.get_ContainerContext: TIWContainerContext;
begin
  result := FContainerImplementation.ContainerContext;
end;

procedure TIWCustomGrid.set_ContainerContext(const AContainerContext: TIWContainerContext);
begin
  if Assigned(ContainerContext) then begin
    ContainerContext.Free;
  end;
  FContainerImplementation.set_ContainerContext(AContainerContext);
end;

function TIWCustomGrid.get_HasTabOrder: boolean;
begin
  result := false;
end;

function TIWCustomGrid.GenerateControlPositions: String;
begin
  result := '';
end;

procedure TIWCustomGrid.RenderCells(AContext: TIWComponent40Context; AGridTag: TIWHTMLTag);
begin
  // Before rendering cells we need to initialize Contianer context
  ContainerContext := IWBaseRenderContext.TIWContainerContext.Create(AContext.WebApplication);
end;

procedure TIWCustomGrid.Freeing(AObject: TObject);
begin
  // DO Nothing
end;

function TIWCustomGrid.ContainerPrefix: string;
begin
  if SupportsInterface(Parent, IIWBaseContainer) then begin
    result := BaseContainerInterface(Parent).ContainerPrefix;
  end else begin
    result := UpperCase(Name);
  end;
end;

{ TIWGridCells }

function TIWGridCells.Add: TIWGridCell;
begin
  Result := TIWGridCell(inherited Add);
end;

procedure TIWGrid.InitControl;
begin
  inherited;
  FCellCollection := TIWGridCells.Create(Self, TIWGridCell);
  // Bypass setter routines
  FColumnCount := 1;
  FRowCount := 1;
  FShowEmptyCells := True;
  // Initialize FCells becouse in .NET these are empty and new size can't be set.
  SetLength(FCells, 1);
  SetLength(FCells[0], 1);

  FScrollToCurrentRow := False;
  FCurrentRow := -1;
end;

procedure TIWGrid.Dispose(ADispose: Boolean); 
begin
  FreeAndNil(FCellCollection);
  inherited;
end;

function TIWGrid.CellExists(const ARow, AColumn: Integer): Boolean;
begin
  if (AColumn >= ColumnCount) or (ARow >= RowCount) or (AColumn < 0) or (ARow < 0) then begin
    raise Exception.Create(RSIndexOutOfBounds);
  end;
  Result := FCells[ARow, AColumn] <> nil;
end;

function TIWGrid.GetCell(const ARow, AColumn: Integer): TIWGridCell;
begin
  if not CellExists(ARow, AColumn) then begin
    FCells[ARow, AColumn] := FCellCollection.Add;
    FCells[ARow, AColumn].FDesignMode := IsDesignMode;
    FCells[ARow, AColumn].Grid := Self;
    // this will copy current Grid font settings to all cells after they were created
    // Later in OnRenderCell the user program can change these settings by loading different values in
    // GridCell.Font
    FCells[ARow, AColumn].Font.Assign(Self.WebFont);
    Invalidate;
  end;
  Result := FCells[ARow, AColumn];
end;

procedure TIWGrid.RenderCells(AContext: TIWComponent40Context; AGridTag: TIWHTMLTag);
var
  i: Integer;
  j: Integer;
  LText: String;
  LBuffer:TIWRenderStream;
  LRowID : string;
  LScrollToCurrentRowCode : string;
begin
  inherited;
  for i := 0 to RowCount - 1 do begin
    with AGridTag.Contents.AddTag('TR') do begin
      for j := 0 to ColumnCount - 1 do begin
        if CellExists(i, j) then begin
          with FCells[i, j] do begin
            if Visible then begin
              if Clickable then begin
                with SubmitLink(Text, IntToStr(i) + '_' + IntToStr(j), False, AContext.Browser) do begin
                  LBuffer:=TIWRenderStream.Create;
                  try
                    Render(LBuffer);
                    LText := LBuffer.Extract(); //TODO: Remove call to Extract
                  finally
                    LBuffer.Free;
                  end;
                  Free;
                end;
                Contents.AddTagAsObject(RenderSelf(Self, i, j, AContext, LText));
              end else begin
                Contents.AddTagAsObject(RenderSelf(Self, i, j, AContext));
              end;
            end else begin
              Contents.AddTagAsObject(RenderSelf(Self, i, j, AContext));
            end;
          end;
        end else begin
          with Contents.AddTag('TD') do begin
            if IsDesignMode or ShowEmptyCells then begin
              Contents.AddText('&nbsp;');
            end else begin
              Contents.AddText('');
            end;
          end;
        end;  // if CellExists
      end;  // for j

      if FScrollToCurrentRow and (i = FCurrentRow) then begin
       {$IFDEF CLR}
        LRowID := Format('%d%d', [TimeStampToMSecs(DateTimeToTimeStamp(Now)), i]);
        {$ELSE}
        LRowID := Format('%d%d', [integer(Self), i]);
        {$ENDIF}
        AddStringParam('ID', LRowID);

        LScrollToCurrentRowCode := iif(AContext.WebApplication.IsPartialUpdate, 'IWTop().', '')
           + 'document.getElementById("' + HTMLName +
          '").scrollTop = ' + iif(AContext.WebApplication.IsPartialUpdate, 'IWTop().', '') + 'document.getElementById("' + LRowID + '").offsetTop - 40;';

        with AContext.PageContext as TIWPageContext40 do begin
          if WebApplication.IsPartialUpdate then begin
            AddToUpdateInitProc(LScrollToCurrentRowCode);
          end else begin
            AddToInitProc(LScrollToCurrentRowCode);
          end;
        end;
      end;
    end;  // with TR tag
  end;  // for
end;

function TIWGrid.GetSubmitParam: String;
begin
  Result := FSubmitParam;
end;

procedure TIWGrid.SetColumnCount(const AValue: Integer);
var
  i: Integer;
  j: Integer;
begin
  if (AValue <> FColumnCount) and (AValue > 0) then begin
    // if not DesignMode then begin
      for i := 0 to High(FCells) do begin
        if AValue < FColumnCount then begin
          for j := ColumnCount - 1 downto AValue do begin
            if Assigned(FCells[i, j]) then begin
              FCellCollection.Delete(FCells[i,j].Index);
            end;
          end;
        end;
        SetLength(FCells[i], AValue);
        if AValue > FColumnCount then begin
          for j := FColumnCount to AValue - 1 do begin
            FCells[i, j] := nil;
          end;
        end;
      end;
    // end;
    FColumnCount := AValue;
    Invalidate;
  end;
end;

procedure TIWGrid.SetRowCount(const AValue: Integer);
var
  i: Integer;
  j: Integer;
begin
  if AValue < 0 then begin
    raise Exception.Create('Invalid RowCount value!');
  end; 

  if AValue <> FRowCount then begin
    // if not DesignMode then begin
      if AValue < FRowCount then begin
        for i := FRowCount - 1 downto AValue do begin
          for j := ColumnCount - 1 downto 0 do begin
            if Assigned(FCells[i, j]) then begin
              FCellCollection.Delete(FCells[i, j].Index);
            end;
          end;
        end;
      end;
      SetLength(FCells, AValue);
      if AValue > FRowCount then begin
        for i := FRowCount to AValue - 1 do begin
          SetLength(FCells[i], FColumnCount);
          for j := 0 to High(FCells[i]) do begin
            FCells[i, j] := nil;
          end;
        end;
      end;
    // end;
    FRowCount := AValue;
    Invalidate;
  end;
end;

procedure TIWGrid.DoCellClick(const ARow, AColumn: Integer);
begin
  if Assigned(OnCellClick) then begin
    OnCellClick(Self, ARow, AColumn);
  end;
end;

procedure TIWGrid.Submit(const AValue: string);
var
  s: string;
  LColumn: Integer;
  LRow: Integer;
begin
  FSubmitParam := AValue;
  s := AValue;
  LRow := StrToInt(Fetch(s, '_'));
  LColumn := StrToInt(s);
  DoCellClick(LRow, LColumn);
end;

procedure TIWGrid.DeleteRow(const ARow: Integer);
var
  i, j: Integer;
begin
  if (ARow < 0) or (ARow >= RowCount) then begin
    raise Exception.Create(RSIndexOutOfBounds);
  end;

  Invalidate;

  for i := High(FCells[ARow]) downto 0 do begin
    if CellExists(ARow, i) then begin
      FCellCollection.Delete(FCells[ARow, i].Index);
    end;
  end;
  for i := ARow to High(FCells) - 1 do begin
    for j := 0 to High(FCells[i]) do begin
      FCells[i, j] := FCells[i + 1, j];
    end;
  end;
  for i := 0 to High(FCells[RowCount - 1]) do begin
    FCells[RowCount - 1, i] := nil;
  end;
  // This will call SetRowCount
  RowCount := RowCount - 1;
end;

procedure TIWGrid.DeleteColumn(const AColumn: Integer);
var
  i: Integer;
  j: Integer;
begin
  if (AColumn < 0) or (AColumn >= ColumnCount) then begin
    raise Exception.Create(RSIndexOutOfBounds);
  end;

  Invalidate;

  for i := 0 to High(FCells) do begin
    for j := AColumn to High(FCells[i]) - 1 do begin
      if (j = AColumn) and CellExists(i, j) then begin
        FCells[i, j].Free;
      end;
      FCells[i, j] := FCells[i, j + 1];
    end;
  end;

  for i := 0 to High(FCells) do begin
    FCells[i, ColumnCount - 1] := nil;
  end;
  // This will call SetColumnCount
  ColumnCount := ColumnCount - 1;
end;

procedure TIWGridCells.Invalidate;
begin
  {$IFNDEF VSNET}
  DoRefreshControl := True;
  if (Owner <> nil) and (Owner is TPlatformControl) and not (csDestroying in TPlatformControl(Owner).ComponentState) then
    (Owner as TControl).Invalidate;
  {$ENDIF}
end;

function TIWGridCells.Owner: TPersistent;
begin
  result := inherited GetOwner;
end;

function TIWGrid.IsValidCell(ACell: TIWGridCell): Boolean;
Var
  LCell: TCollectionItem;
begin
  LCell := FCellCollection.FindItemID(ACell.ID);
  result := LCell = ACell;
end;

procedure TIWGrid.Clear;
Var
  i, j: Integer;
begin
  for i := Low(FCells) to High(FCells) do begin
    for j := Low(FCells[i]) to High(FCells[i]) do begin
      if CellExists(i, j) then begin
        FreeAndNil(FCells[i, j]);
      end;
    end;
  end;
  Invalidate;
end;

procedure TIWGrid.Freeing(AObject: TObject);
Var
  i, j: Integer;
begin
  for i := Low(FCells) to High(FCells) do begin
    for j := Low(FCells[i]) to High(FCells[i]) do begin
      if CellExists(i, j) then begin
        if Cell[i, j].Control = AObject then
          Cell[i, j].Control := nil; 
      end;
    end;
  end;
  Invalidate;
end;

{procedure TIWGrid.SetValue(const AValue: string);
Var
  i, j: Integer;
begin
  for i := Low(FCells) to High(FCells) do begin
    for j := Low(FCells[i]) to High(FCells[i]) do begin
      if CellExists(i, j) then begin
        if SupportsInterface(Cell[i, j].Control, IIWInputControl) then begin
          if InputInterface(Cell[i, j].Control).IsForThisControl(AName) then begin
            InputInterface(Cell[i, j].Control).SetValue(AName, AValue);
          end;
        end;
      end;
    end;
  end;
end;

function TIWGrid.IsForThisControl(AName: string): boolean;
Var
  i, j: Integer;
begin
  result := false;
  for i := Low(FCells) to High(FCells) do begin
    for j := Low(FCells[i]) to High(FCells[i]) do begin
      if CellExists(i, j) then begin
        if SupportsInterface(Cell[i, j].Control, IIWInputControl) then begin
          result := InputInterface(Cell[i, j].Control).IsForThisControl(AName);
        end;
        if result then exit;
      end;
    end;
  end;
end;}

procedure TIWCustomGrid.DoRender;
var
  i: Integer;
begin
  if Assigned(OnRender) then begin
    OnRender(Self);
  end;
  for i := 0 to ComponentCount - 1 do begin
    if SupportsInterface(Components[i], IIWBaseContainer) then begin
      BaseContainerInterface(Component[i]).DoRender;
    end;
  end;
end;

{$IFDEF VSNET}
function TIWCustomGrid.getText: string;
begin
  Result := Caption;
end;

procedure TIWCustomGrid.setText(AValue: string);
begin
  Caption := AValue;
  TIWNETBaseControl(Self).Invalidate;
end;
{$ENDIF}


end.

